/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
*/

/*
  Copyright (C) 2005 Francois Chazal <neptune3k@free.fr>
  Copyright (C) 2006-2007 Eike Hein <hein@kde.org>
*/


#include "image_button.h"
#include "image_button.moc"
#include "settings.h"

#include <tqwhatsthis.h>
#include <tqtimer.h>

#include <kglobalsettings.h>


ImageButton::ImageButton(TQWidget* parent, const char* name, bool translucency) : TranslucentWidget(parent, name, translucency)
{
    state = 0;
    toggle = false;
    pressed = false;
    delay_popup = false;

    popup_menu = NULL;
    popup_timer = NULL;
}

ImageButton::~ImageButton()
{
}

void ImageButton::setToggleButton(bool toggled)
{
    /* Sets the toggling ability. */

    pressed = false;
    toggle = toggled;
}

void ImageButton::setToggled(bool enable)
{
    if (toggle)
    {
        state = 0;

        if (enable)
            pressed = true;
        else
            pressed = false;

        tqrepaint();
    }
}

void ImageButton::setPopupMenu(TQPopupMenu* menu)
{
    popup_menu = menu;
    popup_timer = new TQTimer(this);
    connect(popup_timer, TQT_SIGNAL(timeout()), this, TQT_SLOT(showPopupMenu()));
}

void ImageButton::showPopupMenu()
{
    popup_menu->exec(mapToGlobal(TQPoint(0, height())));
}

void ImageButton::setUpPixmap(const TQString& path, bool use_alpha_mask)
{
    up_pixmap.load(path);
    resize(up_pixmap.size());

    if (up_pixmap.hasAlphaChannel()) setMask(*up_pixmap.mask());

    if (use_alpha_mask)
        setUseTranslucency(true);
    else
        setMask(TQRegion(up_pixmap.rect()));
}

void ImageButton::setOverPixmap(const TQString& path)
{
    over_pixmap.load(path);
}

void ImageButton::setDownPixmap(const TQString& path)
{
    down_pixmap.load(path);
}

void ImageButton::enterEvent(TQEvent*)
{
    state = pressed ? 2 : 1;

    tqrepaint();
}

void ImageButton::leaveEvent(TQEvent*)
{
    state = 0;

    if (popup_timer) popup_timer->stop();

    tqrepaint();
}

void ImageButton::mousePressEvent(TQMouseEvent*)
{
    if (TQWhatsThis::inWhatsThisMode()) return;

    state = 2;

    if (popup_timer) popup_timer->stop();

    tqrepaint();

    if (popup_menu)
    {
        if (delay_popup)
            popup_timer->start(600, true);
        else
            popup_menu->exec(mapToGlobal(TQPoint(0, height())));
    }
}

void ImageButton::mouseReleaseEvent(TQMouseEvent*)
{
    if (TQWhatsThis::inWhatsThisMode()) return;

    // Don't process event if press and release didn't
    // occur within the button.
    if (!state > 0) return;

    state = toggle ? 0 : 1;
    pressed = toggle ? !pressed : false;

    if (popup_timer) popup_timer->stop();

    tqrepaint();

    if (toggle)
        emit toggled(pressed);
    else
        emit clicked();
}

void ImageButton::paintEvent(TQPaintEvent*)
{
    TQPainter painter(this);

    erase();

    if (!useTranslucency())
        painter.fillRect(0, 0, width(), height(), Settings::skinbgcolor());

    switch (state)
    {
        case 0:
            if (pressed)
                painter.drawPixmap(0, 0, down_pixmap);
            else
                painter.drawPixmap(0, 0, up_pixmap);
            break;

        case 1:
            painter.drawPixmap(0, 0, over_pixmap);
            break;

        case 2:
            painter.drawPixmap(0, 0, down_pixmap);
            break;
    }

    painter.end();
}
