/*
    This file is part of the KDE games library
    Copyright (C) 2001-2003 Nicolas Hadacek (hadacek@kde.org)

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef KEXTHIGHSCORE_ITEM_H
#define KEXTHIGHSCORE_ITEM_H

#include <tqvariant.h>
#include <tqnamespace.h>
#include <tqmap.h>
#include <tqvaluevector.h>
#include <kdemacros.h>
class TQWidget;


namespace KExtHighscore
{

//-----------------------------------------------------------------------------
/**
 * This class defines how to convert and how to display
 * a highscore element (such as the score, the date, ...) or a player
 * info (such as the player name, the best score, ...).
 */
class KDE_EXPORT Item
{
 public:
    /**
     * Possible display format.
     * <ul>
     * <li> @p NoFormat : no formatting (default) </li>
     * <li> @p OneDecimal : with one decimal (only for Double) </li>
     * <li> @p Percentage : with one decimal + % (only for Double) </li>
     * <li> @p MinuteTime : MM:SS ie 3600 is 00:00, 1 is 59:59 and 0 is
     *      undefined (only for UInt, Int and Double) </li>
     * <li> @p DateTime : date and time according to locale (only for
     *      DateTime) </li>
     * </ul>
     */
    enum Format { NoFormat, OneDecimal, Percentage, MinuteTime,
		          DateTime };

    /**
     * Possible special value for display format.
     * <ul>
     * <li> @p NoSpecial : no special value ; a null DateTime is replaced by
     *      "--" (default) </li>
     * <li> ZeroNotDefined : 0 is replaced by "--" (only for UInt, Int and
     *      Double) </li>
     * <li> @p NegativeNotDefined : negative values are replaced by "--" (only
     *      for Int and Double) </li>
     * <li> @p DefaultNotDefined : default value is replaced by "--" </li>
     * <li> @p Anonymous : replace the special value ItemBase::ANONYMOUS
     *      by i18n("anonymous") (only for String) </li>
     * </ul>
     */
    enum Special { NoSpecial, ZeroNotDefined, NegativeNotDefined,
                   DefaultNotDefined, Anonymous };

    /**
     * Constructor.
     *
     * @param def default value ; the TQVariant also gives the type of data.
     * Be sure to cast the value to the required type (for e.g. with uint).
     * @param label the label corresponding to the item. If empty, the item
     *              is not shown.
     * @param tqalignment the tqalignment of the item.
     */
    Item(const TQVariant &def = TQVariant::Invalid,
         const TQString &label = TQString(), int tqalignment = TQt::AlignRight);

    virtual ~Item();

    /**
     * Set the display format.
     * @see Format
     */
    void setPrettyFormat(Format format);

    /**
     * Set the special value for display.
     * @see Special
     */
    void setPrettySpecial(Special special);

    /**
     * @return if the item is shown.
     */
    bool isVisible() const    { return !_label.isEmpty(); }

    /**
     * Set the label.
     */
    void setLabel(const TQString &label) { _label = label; }

    /**
     * @return the label.
     */
    TQString label() const { return _label; }

    /**
     * @return the tqalignment.
     */
    int tqalignment() const { return _tqalignment; }

    /**
     * Set default value.
     */
    void setDefaultValue(const TQVariant &value) { _default = value; }

    /**
     * @return the default value.
     */
    const TQVariant &defaultValue() const { return _default; }

    /**
     * @return the converted value (by default the value is left
     * unchanged). Most of the time you don't need to reimplement this method.
     *
     * @param i the element index ("rank" for score / "id" for player)
     * @param value the value to convert
     */
    virtual TQVariant read(uint i, const TQVariant &value) const;

    /**
     * @return the string to be displayed. You may need to reimplement this
     * method for special formatting (different from the standard ones).
     *
     * @param i the element index ("rank" for score / "id" for player)
     * @param value the value to convert
     */
    virtual TQString pretty(uint i, const TQVariant &value) const;

 private:
    TQVariant _default;
    TQString  _label;
    int      _tqalignment;
    Format   _format;
    Special  _special;

    class ItemPrivate;
    ItemPrivate *d;

    static TQString timeFormat(uint);
};

//-----------------------------------------------------------------------------
/**
 * Possible score type.
 * @p Won the game has been won.
 * @p Lost the game has been lost or has been aborted.
 * @p Draw the game is a draw.
 */
enum ScoreType { Won = 0, Lost = -1, Draw = -2 };

/**
 * This class contains data for a score. You should not inherit from
 * this class but reimplement the methods in Highscores.
 */
class KDE_EXPORT Score
{
 public:
    Score(ScoreType type = Won);

    ~Score();

    /**
     * @return the game type.
     */
    ScoreType type() const { return _type; }

    /**
     * Set the game type.
     */
    void setType(ScoreType type) { _type = type; }

    /**
     * @return the data associated with the named Item.
     */
    const TQVariant &data(const TQString &name) const;

    /**
     * Set the data associated with the named Item. Note that the
     * value should have the type of the default value of the
     * Item.
     */
    void setData(const TQString &name, const TQVariant &value);

    /**
     * @return the score value.
     *
     * Equivalent to <pre>data("score").toUInt()</pre>.
     */
    uint score() const { return data("score").toUInt(); }

    /**
     * Set the score value.
     *
     * Equivalent to <pre>setData("score", score)</pre>.
     */
    void setScore(uint score) { setData("score", score); }

    /**
     * @return true if this is the worst possible score (ie the default
     * argument of ScoreItem).
     */
    bool isTheWorst() const;

    /**
     * Comparison operator.
     *
     * @see Manager::isStrictlyLess
     */
    bool operator <(const Score &score);

 private:
    ScoreType  _type;
    TQMap<TQString, TQVariant> _data;

    class ScorePrivate;
    ScorePrivate *d;

    friend class MultiplayerScores;

    friend TQDataStream &operator <<(TQDataStream &stream, const Score &score);
    friend TQDataStream &operator >>(TQDataStream &stream, Score &score);
};

KDE_EXPORT TQDataStream &operator <<(TQDataStream &stream, const Score &score);
KDE_EXPORT TQDataStream &operator >>(TQDataStream &stream, Score &score);

/**
 * This class is used to store and show scores for multiplayer games.
 *
 * Example of use:
 * Initialize the class:
 * <pre>
 * KExtHighscore::MultiScore ms(2);
 * ms.setPlayerName(0, "player 1");
 * ms.setPlayerName(1, "player 2");
 * </pre>
 * At the end of each game, add the score of each players:
 * <pre>
 * KExtHighscore::Score score(KExtHighscore::Won);
 * score.setScore(100);
 * ms.addScore(0, score);
 * score.setType(KExtHighscore::Lost);
 * score.setScore(20);
 * ms.addScore(1, score);
 * </pre>
 */
class KDE_EXPORT MultiplayerScores
{
 public:
    MultiplayerScores();

    ~MultiplayerScores();

    /**
     * Set the number of players and clear the scores.
     */
    void setPlayerCount(uint nb);

    /**
     * Set the name of player.
     */
    void setName(uint player, const TQString &name);

    /**
     * Add the score of player.
     */
    void addScore(uint player, const Score &score);

    /**
     * Clear all scores.
     */
    void clear();

    /**
     * Show scores.
     */
    void show(TQWidget *parent);

 private:
    TQValueVector<uint>  _nbGames;
    TQValueVector<Score> _scores;

    class MultiplayerScoresPrivate;
    MultiplayerScoresPrivate *d;

    friend TQDataStream &operator <<(TQDataStream &stream,
                                    const MultiplayerScores &score);
    friend TQDataStream &operator >>(TQDataStream &stream,
                                    MultiplayerScores &score);
};

KDE_EXPORT TQDataStream &operator <<(TQDataStream &stream, const MultiplayerScores &score);
KDE_EXPORT TQDataStream &operator >>(TQDataStream &stream, MultiplayerScores &score);

} // namespace

#endif
